// Bulk Content Writer JavaScript
class BulkContentWriter {
    constructor() {
        this.topics = [];
        this.isGenerating = false;
        this.generatedArticles = [];
        this.progressTracker = null;
        this.pausedArticles = new Set();
        this.init();
    }

    init() {
        this.loadSavedTopics();
        this.setupEventListeners();
        this.updateUI();
        this.loadNotifications();
    }

    setupEventListeners() {
        // Debounced input handlers for auto-save
        document.getElementById('topicsContainer').addEventListener('input', 
            debounce(() => this.saveTopics(), 1000)
        );
    }

    addTopic() {
        const topicsContainer = document.getElementById('topicsContainer');
        const topicItem = document.createElement('div');
        topicItem.className = 'topic-item';
        topicItem.innerHTML = `
            <div class="topic-header">
                <div class="topic-input">
                    <div class="form-group">
                        <label>Topic/Title</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-format-title"></i>
                            <input type="text" class="topic-title" placeholder="Enter topic or title...">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Custom Prompt (Optional)</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-message-text"></i>
                            <textarea class="topic-prompt" placeholder="Leave empty to use default prompt..."></textarea>
                        </div>
                    </div>
                </div>
                
                <div class="topic-actions">
                    <button onclick="removeTopic(this)" class="btn btn-danger btn-sm">
                        <i class="mdi mdi-delete"></i>
                    </button>
                </div>
            </div>
        `;
        
        topicsContainer.appendChild(topicItem);
        this.saveTopics();
    }

    removeTopic(button) {
        const topicItem = button.closest('.topic-item');
        if (document.querySelectorAll('.topic-item').length > 1) {
            topicItem.remove();
            this.saveTopics();
        } else {
            showAlert('You need at least one topic!', 'error');
        }
    }

    getTopics() {
        const topics = [];
        document.querySelectorAll('.topic-item').forEach(item => {
            const title = item.querySelector('.topic-title').value.trim();
            const prompt = item.querySelector('.topic-prompt').value.trim();
            if (title) {
                topics.push({ title, prompt });
            }
        });
        return topics;
    }

    saveTopics() {
        const topics = this.getTopics();
        localStorage.setItem('bulk_writer_topics', JSON.stringify(topics));
    }

    loadSavedTopics() {
        const saved = localStorage.getItem('bulk_writer_topics');
        if (saved) {
            const topics = JSON.parse(saved);
            // Clear existing topics except first one
            const container = document.getElementById('topicsContainer');
            const firstTopic = container.querySelector('.topic-item');
            firstTopic.querySelector('.topic-title').value = topics[0]?.title || '';
            firstTopic.querySelector('.topic-prompt').value = topics[0]?.prompt || '';
            
            // Add additional topics
            for (let i = 1; i < topics.length; i++) {
                this.addTopic();
                const newTopic = container.querySelector('.topic-item:last-child');
                newTopic.querySelector('.topic-title').value = topics[i].title;
                newTopic.querySelector('.topic-prompt').value = topics[i].prompt;
            }
        }
    }

    async generateAll() {
        const topics = this.getTopics();
        if (topics.length === 0) {
            showAlert('Please add at least one topic!', 'error');
            return;
        }

        if (this.isGenerating) {
            showAlert('Content generation is already in progress!', 'error');
            return;
        }

        this.isGenerating = true;
        this.generatedArticles = [];
        this.pausedArticles.clear();
        this.progressTracker = new ProgressTracker(topics.length);

        // Update UI for generation
        this.updateGenerateButton(false);
        this.clearResults();
        this.createProgressAreas(topics);

        // Generate content for each topic
        for (let i = 0; i < topics.length; i++) {
            if (this.pausedArticles.has(i)) {
                continue;
            }
            await this.generateTopicContent(topics[i], i);
            this.progressTracker.increment();
        }

        this.isGenerating = false;
        this.updateGenerateButton(true);
        showAlert('All content generated successfully!', 'success');
    }

    createProgressAreas(topics) {
        const resultsContainer = document.getElementById('generationResults');
        resultsContainer.innerHTML = '';

        topics.forEach((topic, index) => {
            const articleCard = document.createElement('div');
            articleCard.className = 'article-card';
            articleCard.id = `article-${index}`;
            articleCard.innerHTML = `
                <div class="article-header">
                    <div class="article-title">${topic.title}</div>
                    <div class="article-status">
                        <span class="status-badge status-generating">
                            <i class="mdi mdi-loading spinner"></i>
                            Generating...
                        </span>
                    </div>
                </div>
                <div class="article-content">
                    <div class="progress-bar">
                        <div class="progress-fill" style="width: 0%"></div>
                    </div>
                    <div class="streaming-content" id="content-${index}">
                        <div class="streaming"></div>
                    </div>
                </div>
                <div class="article-footer">
                    <div class="article-stats">
                        <span>Words: <span id="wordcount-${index}">0</span></span>
                        <span>Status: <span id="status-${index}">Generating</span></span>
                    </div>
                    <div class="article-actions">
                        <button class="btn btn-warning btn-sm" onclick="pauseResumeArticle(${index})" id="pause-btn-${index}">
                            <i class="mdi mdi-pause"></i> Pause
                        </button>
                        <button class="btn btn-secondary btn-sm" onclick="regenerateArticle(${index})" id="regenerate-btn-${index}" disabled>
                            <i class="mdi mdi-refresh"></i> Regenerate
                        </button>
                        <button class="btn btn-secondary btn-sm" onclick="copyArticle(${index})" id="copy-btn-${index}" disabled>
                            <i class="mdi mdi-content-copy"></i> Copy
                        </button>
                        <button class="btn btn-success btn-sm" onclick="exportArticle(${index})" id="export-btn-${index}" disabled>
                            <i class="mdi mdi-download"></i> Export
                        </button>
                    </div>
                </div>
            `;
            resultsContainer.appendChild(articleCard);
        });
    }

    async generateTopicContent(topic, index) {
        const contentElement = document.getElementById(`content-${index}`);
        const wordCountElement = document.getElementById(`wordcount-${index}`);
        const statusElement = document.getElementById(`status-${index}`);
        const statusBadge = document.querySelector(`#article-${index} .status-badge`);
        const pauseBtn = document.getElementById(`pause-btn-${index}`);
        const regenerateBtn = document.getElementById(`regenerate-btn-${index}`);
        const copyBtn = document.getElementById(`copy-btn-${index}`);
        const exportBtn = document.getElementById(`export-btn-${index}`);
        
        try {
            // Get generation settings
            const settings = this.getGenerationSettings();
            settings.currentTopic = topic.title;
            
            // Generate content using AI - FIXED: Pass custom prompt properly
            const content = await AIService.generateContent(topic, settings, (chunk, fullContent) => {
                // Check if article is paused
                if (this.pausedArticles.has(index)) {
                    return;
                }
                
                contentElement.innerHTML = this.formatContentForDisplay(fullContent);
                const wordCount = fullContent.split(/\s+/).length;
                wordCountElement.textContent = formatWordCount(wordCount);
                
                // Update progress bar
                const progressFill = contentElement.previousElementSibling.querySelector('.progress-fill');
                if (progressFill) {
                    progressFill.style.width = '100%';
                }
            });

            // Update status and store article
            statusBadge.className = 'status-badge status-completed';
            statusBadge.innerHTML = '<i class="mdi mdi-check-circle"></i> Completed';
            statusElement.textContent = 'Completed';
            
            // Enable buttons
            regenerateBtn.disabled = false;
            copyBtn.disabled = false;
            exportBtn.disabled = false;
            pauseBtn.style.display = 'none';

            this.generatedArticles[index] = {
                title: topic.title,
                content: content,
                wordCount: content.split(/\s+/).length
            };

        } catch (error) {
            console.error('Error generating content:', error);
            statusBadge.className = 'status-badge status-error';
            statusBadge.innerHTML = '<i class="mdi mdi-alert-circle"></i> Failed';
            statusElement.textContent = 'Failed';
            contentElement.innerHTML = `<div class="alert alert-error">Failed to generate content: ${error.message}</div>`;
            
            // Enable regenerate button on error
            regenerateBtn.disabled = false;
        }
    }

    pauseResumeArticle(index) {
        const pauseBtn = document.getElementById(`pause-btn-${index}`);
        const statusElement = document.getElementById(`status-${index}`);
        const statusBadge = document.querySelector(`#article-${index} .status-badge`);
        
        if (this.pausedArticles.has(index)) {
            // Resume article
            this.pausedArticles.delete(index);
            pauseBtn.innerHTML = '<i class="mdi mdi-pause"></i> Pause';
            statusElement.textContent = 'Generating';
            statusBadge.className = 'status-badge status-generating';
            statusBadge.innerHTML = '<i class="mdi mdi-loading spinner"></i> Generating...';
        } else {
            // Pause article
            this.pausedArticles.add(index);
            pauseBtn.innerHTML = '<i class="mdi mdi-play"></i> Resume';
            statusElement.textContent = 'Paused';
            statusBadge.className = 'status-badge status-paused';
            statusBadge.innerHTML = '<i class="mdi mdi-pause"></i> Paused';
        }
    }

    async regenerateArticle(index) {
        const topics = this.getTopics();
        if (index >= topics.length) {
            showAlert('Invalid article index!', 'error');
            return;
        }

        const topic = topics[index];
        const contentElement = document.getElementById(`content-${index}`);
        const wordCountElement = document.getElementById(`wordcount-${index}`);
        const statusElement = document.getElementById(`status-${index}`);
        const statusBadge = document.querySelector(`#article-${index} .status-badge`);
        const pauseBtn = document.getElementById(`pause-btn-${index}`);
        const regenerateBtn = document.getElementById(`regenerate-btn-${index}`);
        const copyBtn = document.getElementById(`copy-btn-${index}`);
        const exportBtn = document.getElementById(`export-btn-${index}`);

        // Reset UI
        contentElement.innerHTML = '<div class="streaming"></div>';
        wordCountElement.textContent = '0';
        statusElement.textContent = 'Generating';
        statusBadge.className = 'status-badge status-generating';
        statusBadge.innerHTML = '<i class="mdi mdi-loading spinner"></i> Generating...';
        pauseBtn.style.display = 'inline-flex';
        pauseBtn.innerHTML = '<i class="mdi mdi-pause"></i> Pause';
        regenerateBtn.disabled = true;
        copyBtn.disabled = true;
        exportBtn.disabled = true;

        // Remove from paused articles if it was paused
        this.pausedArticles.delete(index);

        try {
            // Get generation settings
            const settings = this.getGenerationSettings();
            settings.currentTopic = topic.title;
            
            // Regenerate content - FIXED: Pass custom prompt properly
            const content = await AIService.generateContent(topic, settings, (chunk, fullContent) => {
                if (this.pausedArticles.has(index)) {
                    return;
                }
                
                contentElement.innerHTML = this.formatContentForDisplay(fullContent);
                const wordCount = fullContent.split(/\s+/).length;
                wordCountElement.textContent = formatWordCount(wordCount);
            });

            // Update status
            statusBadge.className = 'status-badge status-completed';
            statusBadge.innerHTML = '<i class="mdi mdi-check-circle"></i> Completed';
            statusElement.textContent = 'Completed';
            
            // Enable buttons
            regenerateBtn.disabled = false;
            copyBtn.disabled = false;
            exportBtn.disabled = false;
            pauseBtn.style.display = 'none';

            this.generatedArticles[index] = {
                title: topic.title,
                content: content,
                wordCount: content.split(/\s+/).length
            };

            showAlert('Article regenerated successfully!', 'success');
        } catch (error) {
            console.error('Error regenerating content:', error);
            statusBadge.className = 'status-badge status-error';
            statusBadge.innerHTML = '<i class="mdi mdi-alert-circle"></i> Failed';
            statusElement.textContent = 'Failed';
            contentElement.innerHTML = `<div class="alert alert-error">Failed to regenerate content: ${error.message}</div>`;
            
            regenerateBtn.disabled = false;
        }
    }

    formatContentForDisplay(content) {
        // Enhanced formatting for better readability - FIXED: Better formatting
        let formatted = content
            .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
            .replace(/\*(.*?)\*/g, '<em>$1</em>')
            .replace(/\n/g, '<br>')
            .replace(/### (.*?)(?=\n|$)/g, '<h3>$1</h3>')
            .replace(/## (.*?)(?=\n|$)/g, '<h2>$1</h2>')
            .replace(/# (.*?)(?=\n|$)/g, '<h1>$1</h1>')
            .replace(/`(.*?)`/g, '<code>$1</code>')
            .replace(/&gt;&gt;&gt; (.*?)(?=\n|$)/g, '<blockquote>$1</blockquote>');

        // Ensure proper paragraph structure
        formatted = formatted.replace(/(<br>\s*){2,}/g, '</p><p>');
        if (!formatted.startsWith('<p>')) {
            formatted = '<p>' + formatted;
        }
        if (!formatted.endsWith('</p>')) {
            formatted = formatted + '</p>';
        }

        return formatted;
    }

    getGenerationSettings() {
        return {
            contentType: document.getElementById('contentType').value,
            tone: document.getElementById('tone').value,
            wordCount: parseInt(document.getElementById('wordCount').value),
            aiPersona: document.getElementById('aiPersona').value,
            creativity: parseInt(document.getElementById('creativity').value),
            research: parseInt(document.getElementById('research').value),
            addins: Array.from(document.querySelectorAll('input[name="addins"]:checked')).map(cb => cb.value)
        };
    }

    getDefaultPrompt(topic, settings) {
        return `Write a comprehensive ${settings.contentType} about "${topic}" in a ${settings.tone} tone. 
        Target approximately ${settings.wordCount} words. Include practical insights, real-world examples, 
        and make it engaging and helpful for readers. Focus on the topic without digressions.`;
    }

    updateGenerateButton(enabled) {
        const button = document.querySelector('button[onclick="generateAll()"]');
        if (enabled) {
            button.innerHTML = '<i class="mdi mdi-robot"></i> Generate All Content';
            button.disabled = false;
        } else {
            button.innerHTML = '<i class="mdi mdi-loading spinner"></i> Generating...';
            button.disabled = true;
        }
    }

    clearResults() {
        document.getElementById('generationResults').innerHTML = '';
    }

    async loadNotifications() {
        try {
            const response = await fetch('get_notifications.php');
            const data = await response.json();
            if (data.success) {
                this.updateNotificationsUI(data.notifications);
            }
        } catch (error) {
            console.warn('Failed to load notifications:', error);
        }
    }

    updateNotificationsUI(notifications) {
        const container = document.getElementById('notificationsContainer');
        if (!notifications || notifications.length === 0) {
            return;
        }

        notifications.forEach(notification => {
            const notificationEl = document.createElement('div');
            notificationEl.className = `notification ${notification.type}`;
            notificationEl.dataset.id = notification.id;
            notificationEl.innerHTML = `
                <div class="notification-icon">
                    <i class="mdi mdi-${this.getNotificationIcon(notification.type)}"></i>
                </div>
                <div class="notification-content">
                    <h4>${notification.title}</h4>
                    <p>${notification.message}</p>
                </div>
                <button class="notification-close" onclick="closeNotification(this)">
                    <i class="mdi mdi-close"></i>
                </button>
            `;
            container.appendChild(notificationEl);
        });
    }

    getNotificationIcon(type) {
        const icons = {
            'success': 'check-circle',
            'error': 'alert-circle',
            'warning': 'alert',
            'info': 'information'
        };
        return icons[type] || 'information';
    }

    updateUI() {
        // Update any dynamic UI elements
    }
}

// Global functions for HTML onclick handlers
function addTopic() {
    bulkWriter.addTopic();
}

function removeTopic(button) {
    bulkWriter.removeTopic(button);
}

function generateAll() {
    bulkWriter.generateAll();
}

function pauseResumeArticle(index) {
    bulkWriter.pauseResumeArticle(index);
}

function regenerateArticle(index) {
    bulkWriter.regenerateArticle(index);
}

function exportAll() {
    if (bulkWriter.generatedArticles.length === 0) {
        showAlert('No content to export! Please generate some content first.', 'error');
        return;
    }

    bulkWriter.generatedArticles.forEach(article => {
        if (article && article.content) {
            exportAsDoc(article.content, article.title);
        }
    });
    
    showAlert('All articles exported successfully!', 'success');
}

function copyArticle(index) {
    const article = bulkWriter.generatedArticles[index];
    if (article && article.content) {
        copyToClipboard(article.content);
    } else {
        showAlert('No content available to copy!', 'error');
    }
}

function exportArticle(index) {
    const article = bulkWriter.generatedArticles[index];
    if (article && article.content) {
        exportAsDoc(article.content, article.title);
    } else {
        showAlert('No content available to export!', 'error');
    }
}

function supportDeveloper() {
    window.open('https://quillfy.com/', '_blank');
}

function getHelp() {
    window.open('https://quillfy.com/', '_blank');
}

// FIXED: Secure API Key saving function
async function saveAPIKey() {
    const apiKeyInput = document.getElementById('apiKey');
    const key = apiKeyInput.value.trim();
    
    if (!key) {
        showNotification('Please enter an API key', 'error');
        return;
    }

    // Add the AIzaSy prefix
    const fullKey = 'AIzaSy' + key;
    
    try {
        const response = await fetch('bulkwriter.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: 'action=save_api_key&api_key=' + encodeURIComponent(fullKey)
        });
        
        const data = await response.json();
        
        if (data.success) {
            showNotification('API key saved successfully!', 'success');
            // Clear the input field for security
            apiKeyInput.value = '';
        } else {
            showNotification(data.message || 'Failed to save API key', 'error');
        }
    } catch (error) {
        console.error('API Key Save Error:', error);
        showNotification('Failed to save API key. Please try again.', 'error');
    }
}

function closeNotification(button) {
    const notification = button.closest('.notification');
    const notificationId = notification.dataset.id;
    
    // Mark as read in database
    fetch('bulkwriter.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'action=mark_notification_read&notification_id=' + notificationId
    });
    
    // Remove from UI
    notification.style.animation = 'slideInRight 0.3s ease reverse';
    setTimeout(() => {
        notification.remove();
    }, 300);
}

function showSettings() {
    document.getElementById('settingsModal').style.display = 'flex';
}

function hideSettings() {
    document.getElementById('settingsModal').style.display = 'none';
}

function saveSettings() {
    const apiKey = document.getElementById('settingsApiKey').value;
    if (apiKey) {
        saveAPIKey();
    }
    hideSettings();
}

// Initialize the bulk writer when the page loads
let bulkWriter;
document.addEventListener('DOMContentLoaded', function() {
    bulkWriter = new BulkContentWriter();
});